; IR remote controller sequencer (Assistant)
; Data is stored in 16 bits, one bite is the data level from decoded IR receiver (bit 15) and the remaining 15 bits (0-14) are for the period at 666.666ns per count (21.85ms total). 
; Longer periods are carried through into the next 16-bits. 


	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1459
	#include p16f1459.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_NSLEEP & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF &_IESO_OFF & _FCMEN_OFF

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _CPUDIV_NOCLKDIV & _USBLSCLK_24MHz & _PLLMULT_3x & _PLLEN_DISABLED & _STVREN_ON & _BORV_HI & _LPBOR_OFF & _LVP_OFF 


; Define variables at memory locations

; Bank 0 RAM
ADDRESS0		equ	H'20'	; address ms byte
ADDRESS1		equ	H'21'	; address
TIMEOUT		equ	H'22'	; wait for a low at record start timeout
OVER			equ	H'23'	; address overflow
EDGE			equ	H'24'	; edge flag
DATA_VAL		equ	H'25'	; data value (low or high)
RECORD_PLAY	equ	H'26'	; record or playback flag	
VALUE_1		equ	H'27'	; delay values 1,2,3
VALUE_2		equ	H'28'
VALUE_3		equ	H'29'
COUNTR_1		equ	H'2A'	; timeout counter

; All Banks RAM
STORE3			equ	H'70'	; used in data aquisition
RD_WRT		equ	H'71'	; issue WRITE (H02) or READ (03)  instruction
COUNT0		equ	H'72'	; counter ms byte duration counter
COUNT1		equ	H'73'	; counter ls byte
TEMPD1		equ	H'74'	; temp A/D store
MEMRY0		equ	H'75'	; memory address MS byte 
MEMRY1		equ	H'76'	; memory address
MEMRY2		equ	H'77'	; memory address LS byte
CLOSED		equ	H'78'	; switch closed flag
TEMP			equ	H'79'	; temporary working
TEMP1			equ	H'7A'	; Q1,2 gate ramp down
TEMP2			equ	H'7B'	; Q1,2 ramp down
DUTY			equ	H'7C'	; 33% duty cycle for PWM
		
; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN
;	org    	4			; not used vector (interrupt vector 0004h)

;***************************************************************************************

; RESET		
; Set ports A & B

MAIN
; initial values
	clrf		RECORD_PLAY	; record or playback flag, initially play	

; set inputs/outputs
	movlw	H'00'		; set all outputs low
	movwf	PORTB
	movwf	PORTC	
	movwf	PORTA
	bsf		PORTB,5	; Memory off 	
; keep CS bar low when power off (at PORTB,7)	
	bsf		PORTC,4	; IR receiver off
	
;USB
	movlb	D'29'
	bcf		UCON,3		; USB off
	bcf		UCON,1
	
; weak pullups off/on
	movlb	D'4'			; bank4	WPUA/B
	clrf		WPUA
	bsf		WPUA,4		; PORTA,4 pullup for switches
	clrf		WPUB

; set I/O
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00111011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00000000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'01000000'
	movwf	TRISC

; option
	movlw	B'00000000'	; weak pullups set via WPUA/B
	movwf	OPTION_REG	; 

; watchdog timer set for 32 seconds
	movlw	B'00011110'
	movwf	WDTCON

; analog inputs, A/D
	movlb	D'3'			; bank3	ANSELA/B/C
	movlw	B'00000000'	; 
	movwf	ANSELA
	movlw	B'00000000' ; 
	movwf	ANSELB
	movlw	B'01000000'	; AN8
	movwf	ANSELC
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100000'	; channel 8
	movwf	ADCON0
	movlw	B'01100000'	; left justified A/D result, fosc /64, Vdd to Vss A/D
	movwf	ADCON1
; oscillator	
	movlw	B'11111100'	; (11111100 for 48MHz; 16MHz x 3PLL)
	movwf	OSCCON	; osc

; timer2 set
	movlb	D'0'			; bank0	
	movlw	D'78'		; modulation at 38kHz, initial setting, adjustable with VR1 when read at playback
	movwf	PR2			; PWM period register

; use prescaler of /4 and 48MHz clock AN8 for VR1, divide by 16 add to 71 for 71 to 86 (41.666kHz to 34.48kHz)
; PR2=82 for 36kHz
; PR2=78 for 38kHz
; PR2=74 for 40kHz

	movlw	B'00000101'
	movwf	T2CON
; timer1 set
	movlw	B'00110001'	; 48MHz/4/8; 1.5MHz counter 666.666ns period
	movwf	T1CON

; PWM set
	movlb	D'12'			; bank12  PWM	
	movlw	B'11100000'		; set PWM mode
	movwf	PWM1CON		; enable PWM1 operation
	clrf		PWM1DCL
	clrf		PWM1DCH		; low output when cleared. (PR2 is 71-86. Duty cycle set at about 32-33%)

; SPI
	movlb	D'04'			; bank4 SPI
	movlw	B'11000000'		; 
	movwf	SSPSTAT
	movlw	B'00000000'
	movwf	SSPCON1		; bit 5 enables SPI when set

; low power regulator
	movlb	D'03'
	bsf		VREGCON,1		; VREGPM regulator in low power during sleep

; ensure the USB is off as it affects the RA0 and RA1 inputs for switches
	movlb	D'29'
	bcf		UCON,3			; usb off

READY_SLEEP
; IOC interrupt on change (low) for RA0,1,4
	movlb	D'7'				; IOC registers
	movlw	B'00010011'
	movwf	IOCAN			; interrupt on negative edge for RA0,1,4
	clrf		IOCAF			; clear IOC flags
	movlb	D'0'				; bank0	

; no interrupt routine
	bcf		INTCON,GIE		; ensure global interrupt is  off

PREP_SLEEP ; prepare to sleep

	clrf		CLOSED		; switch closed flag
	bsf		INTCON,PEIE	; peripheral interrupt enable
; power off IR receiver
	movlb	D'2'
	bsf		LATC,4			; gate of Q1 high
; power off Indicator LED
	bcf		LATC,3	
; power off IR LED
	bcf		LATC,5
; power off Memory
	bcf		LATB,7			; first CS bar low
	bsf		LATB,5			; power off (gate of Q2 high)

; SPI disable
	movlb	D'04'			; bank4 SPI
	clrf		SSPCON1		; bit 5 enables SPI when set

	movlb	D'1'				; bank1	TRISA/B/C
	movlw	B'00000000'		; I/O (RB outputs)
	movwf	TRISB			; PORTB,4 output

	movlb	D'2'
	bcf		LATB,6			; SCK low
	bcf		LATB,4			; SPI output low
	bcf		LATC,7			; SP0 output low
			
; prepare switches
	bcf		LATC,0			; Switches 1,3,4
	bcf		LATC,1			; Switches 2,5,6
	bcf		LATC,2			; Switches 7,8,9
	movlb	D'0'
	movf	PORTA,w		; read switches at RA0,RA1,RA4
	iorlw	B'11101100'		; if all 1's then none are pressed (low)
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	SWITCHES

; if zero then none are pressed 
; play mode
	clrf		RECORD_PLAY	; play mode

; allow IOC interrupt to wake from sleep

	bsf		INTCON,IOCIE	; interrupt on change for switches to allow a wakeup from sleep
	SLEEP
	nop	
; stop interrupt on change
	bcf		INTCON,IOCIE	; no interrupt on change
	bcf		INTCON,IOCIF	; flag off

; wait for stable oscillator
	movlb	D'01'				; bank 1
HF1
	btfss	OSCSTAT,HFIOFS	; HF internal oscillator is stable
	goto	HF1					; not stable yet
	btfss	OSCSTAT,HFIOFR	; HF internal oscillator ready
	goto	HF1					; not ready yet
	btfss	OSCSTAT,PLLRDY	; phase lock loop ready check
	goto	HF1
;	movlb	D'0'					; bank 0

; SPI enable
	movlb	D'1'				; bank1	TRISA/B/C
	movlw	B'00010000'		; I/O (RB outputs)
	movwf	TRISB			; PORTB,4 set as an input (for SPI)

	movlb	D'04'			; bank4 SPI
	clrf		SSPCON1		; bit 5 enables SPI when set
	nop
	bsf		SSPCON1,5		; enable SPI
	movlb	D'0'

SWITCHES
; start a period for switch check in record mode
	movlw	D'87'			; ~115ms per value so ~10s for 87
	movwf	COUNTR_1	

; if in record mode, check for a closed switch for a period before cancelling
SW	
	call		DELAY
	call		CK_SW			; find closed switch
	brw						; branch to PC counter location plus W
; Branch Table
	goto	NO_SW			; no switch
	goto	SWITCH1
	goto	SWITCH2
	goto	SWITCH3
	goto	SWITCH4
	goto	SWITCH5
	goto	SWITCH6
	goto	SWITCH7
	goto	SWITCH8
	goto	SWITCH9

NO_SW ; if in record mode wait for a switch closure for a time period. 
	clrf		CLOSED		; switch closed flag
	btfss	RECORD_PLAY,0
	goto	READY_SLEEP	; if zero return to sleep
; record mode so check switches for a period before cancelling record mode

; count waiting period for mode open before another switch is pressed

	decfsz	COUNTR_1,f
	goto	SW	

	goto	PREP_SLEEP	; if zero return to sleep

SWITCH9
; if switch flag is set then bypass increment 	
	btfss	CLOSED,0		; switch closed flag

;Toggle RECORD_PLAY,0 flag. If set then recording begins
	incf		RECORD_PLAY,f	; if set then set up for record then wait for a switch (1-8)
	bsf		CLOSED,0		; set flag so no repeated toggle of RECORD_PLAY
; check for record or play
	btfss	RECORD_PLAY,0
	goto	PLAY1

	bsf		PORTC,3		; indicator LED on
; clear memory counter
	movlw	D'82'			; switch pressed period counter (when pressed for that time, clear the memory). 10seconds
	movwf	TEMP
WAIT_OPEN9
	decfsz	TEMP,f
	goto	CONT1
	goto	CLEAR_MEM	; clear all memory 
CONT1	
; wait for switch 9 to open, then wait for a switch closure
	call		DELAY
; wait for RA0 to go high
	btfss	PORTA,0
	goto	WAIT_OPEN9
 	call		DELAY
; wait for RA0 to go high
	btfss	PORTA,0
	goto	WAIT_OPEN9
	clrf		CLOSED		; switch closed flag 
	goto	SWITCHES		; check switches	

PLAY1
; IR receiver off
	movlb	D'2'
	bsf		LATC,4			; high to switch off Q1 to power down infrared receiver
	bcf		LATC,3			; low for indicator LED off
	movlb	D'0'
	goto	WAIT_OPEN9

SWITCH1
; start up at memory address starting at 000000. For switch 1 (S1)
	clrf		MEMRY0
	clrf		MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH2
; start address for memory for S2
; start up at memory address starting at H4000.
	clrf		MEMRY0
	movlw	H'40'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH3
; start address for memory for S3
; start up at memory address starting at H8000.
	clrf		MEMRY0
	movlw	H'80'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH4
; start address for memory for S4
; start up at memory address starting at HC000.
	clrf		MEMRY0
	movlw	H'C0'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH5
; start address for memory for S5
; start up at memory address starting at H10000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'00'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH6
; start address for memory for S6
; start up at memory address starting at H14000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'40'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH7
; start address for memory for S7
; start up at memory address starting at H18000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'80'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	PLAY_REC
;
SWITCH8
; start address for memory for S8
; start up at memory address starting at H1C000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'C0'
	movwf	MEMRY1
	clrf		MEMRY2
;	goto	PLAY_REC

PLAY_REC
; check if play or record mode
	btfss	RECORD_PLAY,0
	goto	PLAY

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
RECORD

; power up IR receiver
; ramp gate pulses slowly to charge bypass capacitor, preventing a reset on microcontroller

	movlw	D'200'
	movwf	TEMP2				; cycles

	movlb	D'2'					; bank for latch
LOOPX
	movf	TEMP2,w
	movwf	TEMP1
	bcf		LATC,4				; Q1 gate low
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	bsf		LATC,4				; gate off (high)

REPEAT1
	decfsz	TEMP1,f
	goto	REPEAT1			; gate off duty cycle gradually decreases
	decfsz	TEMP2,f
	goto	LOOPX				; gate has ramped toward 0
; now apply gate fully on
	bcf		LATC,4
	movlb	D'0'					; required for bank change to standard '0'

; set switch lines at RC0,1,2 all low. So any switch pressed can be detected
	movlb	D'2'					; bank for latch
	bcf		LATC,0
	bcf		LATC,1
	bcf		LATC,2
	movlb	D'0'					; bank 0

; wait for switch off
WAIT_OFF
	clrwdt						; watchdog cleared
	call		DELAY
	movf	PORTA,w			; read switches
	iorlw	B'11101100'			; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF
	call		DELAY
	movf	PORTA,w			; read switches
	iorlw	B'11101100'			; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF

; power on Memory
; memory powers up in sequential mode

	call		MEM_UP			; power up memory

	call		DELAY

; clear record memory
	movlw	D'02'				; write instruction
	movwf	RD_WRT
	call		READ_WRITE		; issue write instruction
; memory counter address cleared
	clrf		ADDRESS0			; address 0 for each memory block
	clrf		ADDRESS1	
	call		MEM_ADDRESS 	; memory address start point (address dependent on switch)
	clrf		COUNT0			; data memory cleared
	clrf		COUNT1
	clrf		DATA_VAL			; data cleared
		
REWRITE_SX
	call		DATA_WRITE 		; write to memory

; check ADDRESS0,1 and stop clearing memory when it reaches H2000 (ie has gone through that addresses block) 16k x 8 for eack switch
	movf	ADDRESS0,w
	xorlw	H'20'				; end address H2000 (counts 8k x16)
	btfss	STATUS,Z
	goto	REWRITE_SX
	movf	ADDRESS1,w
	btfss	STATUS,Z
	goto	REWRITE_SX

; setup a write instruction and memory start address 
; setup memory for write
; memory set with write mode and address zero
; write mode

	movlw	D'02'				; write instruction
	movwf	RD_WRT
	call		READ_WRITE		; issue write instruction

; memory counter address cleared (counts at half rate as incremented each 16 bits not 8 bits)
	clrf		ADDRESS0			; address 0
	clrf		ADDRESS1
	call		MEM_ADDRESS		 ; memory address start point

; set switch lines at RC0,1,2 all low. So any switch pressed can be detected
	movlb	D'2'					; bank for latch
	bcf		LATC,0
	bcf		LATC,1
	bcf		LATC,2
	movlb	D'0'					; bank 0

; flash LED off and then on
	bcf		PORTC,3
	call		DELAY
	call		DELAY
	bsf		PORTC,3	

; wait for a low at RA5 to start recording
	movlw	D'229'
	movwf	TIMEOUT			; timeout period 229 x 43.69ms =~10seconds

WAIT_START

; if a switch is pressed, then exit

; is a Switch pressed? If PORTA0,1 or 4 are low then a switch is pressed.. Then  goto PREP_SLEEP 

	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF1

	btfss	PIR1,TMR1IF		; overflow flag 
	goto	LOW_5
	bcf		PIR1,TMR1IF		; overflow cleared 
	decfsz	TIMEOUT,f
	goto	LOW_5
	goto	PREP2			; prepare for sleep
LOW_5
	btfsc	PORTA,5
	goto	WAIT_START	

; low at  RA5, and store in data value
	bcf		DATA_VAL,5

; load timer 1 with initial value of H8000 so overflows at 1/2 full count since only 15 bits are used for counter. ms bit is for IR receiver level 
 	bcf		T1CON,0		; stop timer 1
	bcf		PIR1,TMR1IF		; overflow flag cleared
	movlw	H'80'
	movwf	TMR1H
	clrf		TMR1L
	bsf		T1CON,0		; start timer1

NXT_VALUE
; if level changes or if timer1 overflows (PIR1,TMR1IF) , then write timer and level to memory. Stop if ADDRESS0,1 reaches limit or if Switch is pressed

; check ADDRESS0,1 and stop when it reaches H2000 (ie has gone through that addresses block) 16k x 8bit wide for each switch or 8k x 16bit wide

	movf	ADDRESS0,w
	xorlw	H'20'				; end address H2000 for 16bit wide
	btfsc	STATUS,Z
	goto	PREP2				; end

; drive LED when low
	btfss	DATA_VAL,5
	bsf		PORTC,3
	btfsc	DATA_VAL,5		; LED off when data high
	bcf		PORTC,3

MORE_MEM ; more memory left
; has timer1 overflowed?

	btfss 	PIR1,TMR1IF		; check overflow flag at TMR1IF 
	goto	SWx
; set COUNT values
	movlw	H'7F'
	movwf	COUNT0
	movlw	H'FF'
	movwf	COUNT1

	bcf		T1CON,0	
	goto	STOX

SWx
; is a Switch pressed? If PORTA0,1 or 4 are low then a switch is pressed.. Then  goto PREP_SLEEP 
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF1

COMP1
; compare PORTA,5 with data value. If different then store (STO1)
	movf	PORTA,w
	andlw	B'00100000'	
	movwf	TEMP				; store
	xorwf	DATA_VAL,w		; bit 5 kept
	btfsc	STATUS,Z			; compare  current data value
	goto	MORE_MEM

	clrwdt						; watchdog cleared

; stop timer and store
	bcf		T1CON,0
	movf	TMR1H,w
	movwf	COUNT0		; 15 ls bits are the data value period (666.66ns per count) (21.845ms total)
	movf	TMR1L,w
	movwf	COUNT1
; reload timer
STOX
	bcf		PIR1,TMR1IF		; overflow flag cleared
	movlw	H'80'
	movwf	TMR1H
	movlw	D'01'
	movwf	TMR1L
	bsf		T1CON,0		; start timer1

; DATA_VAL has level data for memory write 
	comf	DATA_VAL,f		; use complementary value for data as IR receiver output is low for a signal, high for no signal
	call		DATA_WRITE 	; write to memory
; update data
	movf	TEMP,w
	movwf	DATA_VAL		; data value renewed
	goto	NXT_VALUE

PREP2; prepare for sleep
; flash LED
	bsf		PORTC,3
	call		DELAY

	movlb	D'2'
	bsf		LATC,4			; Q1 gate low for IR receiver off
	movlb	D'0'
	goto	PREP_SLEEP	; prepare for sleep

; wait for switch off
WAIT_OFF1
	clrwdt					; watchdog cleared
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high, switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF1
	call		DELAY
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high, switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF1
	goto	PREP2			; prepare for sleep

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

PLAY
; power on Memory
;  memory powers up in sequential mode

	call		MEM_UP			; power up memory, returns in bank 0

; read VR1 for PWM frequency adjustment (no software reversal for reading is needed as VR1 is arranged with fully anticlockwise is 5V and fully clockwise is 0V so frequency is higher with clockwise movement)
	bsf		PORTC,3			; VR1 powered + LED indicator on

	call		DELAY				; time for VR1 wiper input to rise as bypassed with a 100nF capacitor
	call		ACQUIRE_AD 		; Get digital value. Result in TEMPD1
	bcf		PORTC,3			; indicator and VR1 off
	lsrf		TEMPD1,f			; /2
	lsrf		TEMPD1,f			; /4
	lsrf		TEMPD1,f			; /8
	lsrf		TEMPD1,w			; /16
; divide by 16 add to 71 for 71 to 86 (41.66kHz to 34.4kHz)
	addlw	D'71'
; PWM period register
	movwf	PR2					; PR2=82 for 36kHz; PR2=78 for 38kHz; PR2=75 for 40kHz

; Calculate the 32% duty value for PWM
; divide PR2 by 2. Then take 1/8th of PR2 and 1/16th of PR2 away
	lsrf		PR2,w
	movwf	DUTY				; 50%
	lsrf		DUTY,w				; /4
	movwf	TEMP
	lsrf		TEMP,f				; /8 (12.5%)
; take /8 from DUTY
	movf	TEMP,w
	subwf	DUTY,f				; DUTY/2 - /8
	lsrf		TEMP,w				; /16 (6.25%)
;take /16 from DUTY
	subwf	DUTY,f				; approx 32% of PR2 (50% - 12.5% - 6.25% = 31.25% ) 

; set switch lines at RC0,1,2 all low. So any switch pressed can be detected
	movlb	D'2'					; bank for latch
	bcf		LATC,0
	bcf		LATC,1
	bcf		LATC,2
	movlb	D'0'					; bank 0

; wait until switch is off
WAIT_OFF3
	clrwdt					; watchdog cleared
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF3
	call		DELAY
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF3

; setup a read instruction and memory start address 
; setup memory for read
; memory set with read mode and address
	movlw	D'03'				; read instruction
	movwf	RD_WRT
	call		READ_WRITE		; issue read instruction

; memory counter address cleared (counts at half rate as incremented each 16 bits not 8 bits)
	clrf		ADDRESS0			; address 0
	clrf		ADDRESS1
; write start memory value to memory for starting of stored sequences (memory value loaded at SWITCHx above)
	call		MEM_ADDRESS		 ; memory address start point
LOOP_READ
	call		RD_DATA			; data in COUNT0,1 Ms bit of COUNT0 is the data level (high for off, low for PWM)	

; if COUNT0,1 are both 0 stop play
	movf	COUNT0,w
	btfss	STATUS,Z
	goto	CONT_DATA
	movf	COUNT1,w
	btfsc	STATUS,Z
	goto	PREP1				; both zero

CONT_DATA
; LOOP read data and start PWM pulses for a high (no PWM for a low) and subtract period data from FFFF for timer interrupt	
; extract the ms bit of COUNT0 for data and clear the bit (bit7) 

	btfss	COUNT0,7
	goto	LOW_PWM
; PWM output 
	bsf		PORTC,3		; LED on
	clrf		TMR2
	movlb	D'12'			; bank 12  PWM	
	movf	DUTY,w			; 31.25% duty calculated from PR2 value
	movwf	PWM1DCH		; duty cycle 
	movlb	D'0'				; bank0	
	goto	TIMR
LOW_PWM
; PWM output low
	bcf		PORTC,3		; LED off
	movlb	D'12'			; bank 12  PWM	
	clrf		PWM1DCH		; duty cycle 
	movlb	D'0'				; bank0	
TIMR

	bcf		COUNT0,7		; clear level data bit
; subtract the 15 bit timer from FFFF. Load timer1 and use data level until timer1 times out

	bcf		T1CON,TMR1ON	; timer off
	movf	COUNT1,w		; low byte
	sublw	H'FF'			; low byte subtract 
	movwf	TMR1L			; timer 1 low byte
	movf	COUNT0,w		; high byte
	btfss   	STATUS,C    	 	; carry check
  	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	TMR1H	
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared

	clrwdt					; watchdog cleared

; when  ADDRESS0,1 ended, stop play
; check ADDRESS0,1 and stop when it reaches H2000 (ie has gone through that addresses block) 16k for eack switch
	movf	ADDRESS0,w
	xorlw	H'20'				; end address H2000
	btfsc	STATUS,Z
	goto	PREP1

; Switch off play if switch is pressed

; is a Switch pressed? If PORTA0,1 or 4 are low then a switch is pressed.. Then  goto PREP_SLEEP 
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfsc	STATUS,Z
	goto	WT1

; wait for switch off
WAIT_OFF2
	clrwdt					; watchdog cleared
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF2
	call		DELAY
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFF2
	goto	PREP1			; prepare for sleep

; wait for timer
WT1
	btfss	PIR1,TMR1IF		; when set read next data
	goto	WT1
	bcf		PIR1,TMR1IF	
	goto	LOOP_READ	; next data	

PREP1; prepare for sleep, shut off PWM
; PWM output low
	bcf		PORTC,3		; LED off
	movlb	D'12'			; bank12  PWM	
;	clrf		PWM1DCL
	clrf		PWM1DCH		; duty cycle 
	movlb	D'0'				; bank0	
	goto	PREP_SLEEP

; >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

CLEAR_MEM ; clear memory 

; memory on	

	call		MEM_UP		; power up memory, returns in bank 0	

	call		DELAY

; setup memory for write
; memory set with write mode and address zero
	movlw	D'02'				; write instruction
	movwf	RD_WRT
	call		READ_WRITE		; issue write instruction
; clear data for memory write
	clrf		DATA_VAL			; data value is required level (0 or 1) in normal data mode when storing or recalling IR code
	clrf		COUNT0			; 15 ls bits are the data value period (1us per count) (32.768ms)
	clrf		COUNT1

; check if S1-S8 is also pressed
	call		CK_SW			; find closed switch
	brw						; branch to PC counter location plus W
	goto	READY_SLEEP	; if zero return to sleep
	goto	CLR_SWITCH1
	goto	CLR_SWITCH2
	goto	CLR_SWITCH3
	goto	CLR_SWITCH4
	goto	CLR_SWITCH5
	goto	CLR_SWITCH6
	goto	CLR_SWITCH7
	goto	CLR_SWITCH8

; Clear all memory since no extra switch is pressed
CLR_SWITCH9 ; all memory
; Indicator LED on
	bsf		PORTC,3

; start up at memory address starting at 000000.
	clrf		MEMRY0
	clrf		MEMRY1
	clrf		MEMRY2
	call		MEM_ADDRESS ; memory address start point

; memory counter address cleared (counts at half rate as incremented each 16 bits not 8 bits)
	clrf		ADDRESS0			; address 0
	clrf		ADDRESS1
	clrf		OVER				; overflow flag

REWRITE_ALL
	call		DATA_WRITE 		; write to memory

; check ADDRESS0,1 and stop clearing memory when it reaches H20000 (ie has gone through all addresses). Overflow flag is at H2
	movf	OVER,w
	xorlw	H'02'
	btfss	STATUS,Z			; overflow when =2, then ended
	goto	REWRITE_ALL
	movlw	D'1'
	call		FLASH				; end acknowledgement
	goto	END_CLEARED

;..............................

CLR_SWITCH1
; Flash Indicator LED once
	movlw	D'1'
	call		FLASH		; flash indicator LED

; start address for memory for S1
; start up at memory address starting at 000000.
	clrf		MEMRY0
	clrf		MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH2
; Flash Indicator LED twice
	movlw	D'2'
	call		FLASH		; flash indicator LED

; start address for memory for S2
; start up at memory address starting at H4000.
	clrf		MEMRY0
	movlw	H'40'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH3
; Flash Indicator LED three times
	movlw	D'3'
	call		FLASH

; start address for memory for S3
; start up at memory address starting at H8000.
	clrf		MEMRY0
	movlw	H'80'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH4
; Flash Indicator LED 4 times
	movlw	D'4'
	call		FLASH

; start address for memory for S4
; start up at memory address starting at HC000.
	clrf		MEMRY0
	movlw	H'C0'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH5
; Flash Indicator LED 5 times
	movlw	D'5'
	call		FLASH

; start address for memory for S5
; start up at memory address starting at H10000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'00'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH6
; Flash Indicator LED 6 times
	movlw	D'6'
	call		FLASH

; start address for memory for S6
; start up at memory address starting at H14000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'40'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH7
; Flash Indicator LED 7 times
	movlw	D'7'
	call		FLASH

; start address for memory for S7
; start up at memory address starting at H18000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'80'
	movwf	MEMRY1
	clrf		MEMRY2
	goto	CYC_CLR_MEM

CLR_SWITCH8
; Flash Indicator LED 8 times
	movlw	D'8'
	call		FLASH

; start address for memory for S8
; start up at memory address starting at H1C000.
	movlw	H'01'
	movwf	MEMRY0
	movlw	H'C0'
	movwf	MEMRY1
	clrf		MEMRY2
;	goto	CYC_CLR_MEM

CYC_CLR_MEM	

; wait for switch off

WAIT_OFFMEM
	clrwdt					; watchdog cleared
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFFMEM
	call		DELAY
	movf	PORTA,w		; read switches
	iorlw	B'11101100'		; if all high switches are off
	xorlw	B'11111111'
	btfss	STATUS,Z
	goto	WAIT_OFFMEM

; when switch is off then clear memory

	call		MEM_ADDRESS ; memory address start point
; memory counter address cleared
	clrf		ADDRESS0			; address 0 for each memory block
	clrf		ADDRESS1	
	clrf		OVER				; overflow flag
REWRITE_S
	call		DATA_WRITE 		; write to memory

; check ADDRESS0,1 and stop clearing memory when it reaches H2000 (ie has gone through that addresses block) 16k for eack switch
	movf	ADDRESS0,w
	xorlw	H'20'				; end address H2000
	btfss	STATUS,Z
	goto	REWRITE_S
	movf	ADDRESS1,w
	btfss	STATUS,Z
	goto	REWRITE_S

END_CLEARED ; (from all cleared via S9 only)

; clear RECORD_PLAY flag. Then toggled at SWITCH9 so is in starting record mode.
	clrf		RECORD_PLAY	; clear so in PLAY mode initially
	goto	SWITCH9		; start of record mode

; *****************************************************************************************
; Subroutines
; Flash indicator
FLASH	
	movwf	TEMP		; flash number
FLASHX
	bcf		PORTC,3	; LED off
	movlw	D'255'		; delay extension
	call		DELX
	bsf		PORTC,3	; LED on
	movlw	D'255'
	call		DELX
	decfsz	TEMP,f
	goto	FLASHX
	bcf		PORTC,3	; off for return
	return

;........................................................................................................
;delay 48MHz clock
DELAY		movlw	D'100'		; 100ms 
DELX		movwf	VALUE_3	; 1ms per value
DELT_1		movlw	D'16'		; set delay period 
			movwf	VALUE_1	; VALUE_1 = w
			movlw	D'250'		; set delay period value 2 
LP_1		movwf	VALUE_2		; VALUE_2 = w
LP_2		decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
			goto 	LP_2
			decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
			goto	LP_1
			decfsz	VALUE_3,f
			goto	DELT_1	
			return	
;.......................................................................................................
; read switch closure
CK_SW

;  RC0 low RC2,1 high.
	movlb	D'2'
	bsf		LATC,2
	bcf		LATC,0
	bsf		LATC,1
	movlb	D'0'
	movlw	D'5'				; 5ms for inputs to settle
	call		DELX
; If RA0 low then S1
	btfss	PORTA,0		; if low then S1
	retlw	D'1'				; SWITCH1
; If RA4 low then S4
	btfss	PORTA,4		; if low then S4
	retlw	D'4'				; SWITCH4
; If RA1 low then S3 	
	btfss	PORTA,1		; if low then S3
	retlw	D'3'				; SWITCH3
;.....................................................................................................

;  RC1 low RC2,0 high.
	movlb	D'2'
	bsf		LATC,2
	bsf		LATC,0
	bcf		LATC,1
	movlb	D'0'
	movlw	D'5'				; 5ms
	call		DELX
; If RA0 low then S2
	btfss	PORTA,0		; if low then S2
	retlw	D'2'				; SWITCH2
; If RA4 low then S6
	btfss	PORTA,4		; if low then S6
	retlw	D'6'				; SWITCH6
; If RA1 low then S5 	
	btfss	PORTA,1		; if low then S5
	retlw	D'5'				; SWITCH5

;...................................................................................................
;  RC2 low RC0,1 high.
	movlb	D'2'
	bcf		LATC,2
	bsf		LATC,0
	bsf		LATC,1
	movlb	D'0'
	movlw	D'5'				; 5ms
	call		DELX
; If RA4 low then S8
	btfss	PORTA,4		; if low then S8
	retlw	D'8'				; SWITCH8
; If RA1 low then S7 	
	btfss	PORTA,1		; if low then S7
	retlw	D'7'				; SWITCH7
; If RA0 low then S9
	btfss	PORTA,0		; if low then S9
	retlw	D'9'				; SWITCH9
	retlw	D'0'				; no switch

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; subroutine to wait for A/D conversion (A/D switched on only for conversion, then off to save power)
ACQUIRE_AD
	movlb	D'1'			; bank1	ADCON0
	bsf		ADCON0,ADON	; A/D on

; wait >8us to charge input capacitance/ 
	movlw	D'35'
	movwf	STORE3
WAIT2C1
	decfsz	STORE3,f
	goto	WAIT2C1	
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	
	movf	ADRESH,w
	movwf	TEMPD1		; ls

	bcf		ADCON0,ADON	; A/D off
	movlb	D'0'		; bank0
	return

; subroutines
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
READ_WRITE; read or write instruction RD_WRT = 02 for write, 03 for read
	movlb	D'2'
	bsf		LATB,7	; memory CS off
	nop
	nop
	nop
	bcf		LATB,7	; CS bar for starting up memory chip
	nop

;  issue WRITE instruction (H02)  or READ (03) then ADDRESS (00),

	movlb	D'04'		; bank 4
	movf	RD_WRT,w		
	movwf	SSPBUF
FLAGACLR	
	btfss	SSPSTAT,BF
	goto	FLAGACLR
	movlb	D'0'			; bank 0
	return

;..........................................................................................................................................................
MEM_ADDRESS ; memory address
	movlb	D'04'		; bank 4
; issue memory address (000000) (24-bit address) for 1024byte memory 
; 8-bits ms byte
	movf	MEMRY0,w	
	movwf	SSPBUF
FLAGBCLR	
	btfss	SSPSTAT,BF
	goto	FLAGBCLR
; 16 bits
	movf	MEMRY1,w	
	movwf	SSPBUF
FLAGCCLR	
	btfss	SSPSTAT,BF
	goto	FLAGCCLR
; 24 bits ls byte
	movf	MEMRY2,w	
	movwf	SSPBUF
FLAGDCLR	
	btfss	SSPSTAT,BF
	goto	FLAGDCLR
	movlb	D'0'			; bank 0	
	return
; ..........................................................................................................................................................

DATA_WRITE ; write to memory
; issue data over 16 bits

; get level
	bcf		COUNT0,7		; clear for data bit
	btfsc	DATA_VAL,5
	bsf		COUNT0,7		; set  for data if set
	movf	COUNT0,w		; counter	ms byte	
	movlb	D'4'				; bank4
	movwf	SSPBUF
FLAGA	
	btfss	SSPSTAT,BF
	goto	FLAGA
	movf	COUNT1,w		; counter	ls byte	
	movwf	SSPBUF
FLAGB
	btfss	SSPSTAT,BF
	goto	FLAGB
; increment ADDRESS0,1
	movlb	D'0'				; bank 0
	movlw	D'01'
	addwf	ADDRESS1,f	; ls byte address plus 1
	btfsc	STATUS,C
	incfsz	ADDRESS0,f	; increase address if carry on overflow
	return
	incf		OVER,f			; overflow 
	return

;.............................................................................................

; read Data from memory
; get 16-bit value. Bit 15 is data level
RD_DATA

; enable SPI
	movlb	D'04'			; bank 4
	clrf		SSPBUF		; load with anything
FLAGAR	
	btfss	SSPSTAT,BF
	goto	FLAGAR
	movf	SSPBUF,w		; memory value
	movwf	COUNT0		; ms byte	
	clrf		SSPBUF		; load with anything
FLAGBR	
	btfss	SSPSTAT,BF
	goto	FLAGBR
	movf	SSPBUF,w
	movwf	COUNT1			; ls byte	

; increment ADDRESS0,1
	movlb	D'0'					; bank 0
	movlw	D'01'
	addwf	ADDRESS1,f		; ls byte address plus 1
	btfsc	STATUS,C
	incf		ADDRESS0,f		; increase address if carry on overflow
	return

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; power up memory
; ramp gate pulses slowly to charge capacitor
MEM_UP
	movlw	D'20'
	movwf	TEMP2				; cycles

	movlb	D'2'
	bcf		LATB,7				; memory CS bar low

LOOPZ
	movf	TEMP2,w
	movwf	TEMP1
	bcf		LATB,5				; Q2 gate low
	bsf		LATB,5				; gate off (high)

REPEATZ
	decfsz	TEMP1,f
	goto	REPEATZ			; gate off duty cycle gradually decreases
	decfsz	TEMP2,f
	goto	LOOPZ				; gate has ramped toward 0
; now apply gate fully on
	bcf		LATB,5
	movlb	D'0'					; bank change to  '0'
	return

	end

